import SwiftUI
@_spi(Experimental) import MapboxMaps

struct ColorThemeExample: View {
    enum Theme: String {
        case `default`
        case pink
        case monochrome
    }

    @State private var theme: Theme = .pink
    @State private var panelHeight: CGFloat = 0
    @State private var atmosphereUseTheme = true
    @State private var circleUseTheme = true

    var body: some View {
        Map(initialViewport: .camera(center: .init(latitude: 40.72, longitude: -73.99), zoom: 2, pitch: 45)) {
            switch theme {
            case .default:
                EmptyMapContent()
            case .pink:
                ColorTheme(base64: pinkTheme)
            case .monochrome:
                ColorTheme(uiimage: monochromeTheme)
            }

            Atmosphere()
                .color(.green)
                .colorUseTheme(atmosphereUseTheme ? .default : .none)

            TestLayer(id: "blue-layer", radius: 2, color: .blue, coordinate: .init(latitude: 40, longitude: -104), useTheme: circleUseTheme)

        }
        .mapStyle(.streets) /// In standard style it's possible to provide custom theme using `.standard(themeData: "base64String")`
        .additionalSafeAreaInsets(.bottom, panelHeight)
        .ignoresSafeArea()
        .overlay(alignment: .bottom) {
            VStack(alignment: .center) {
                Group {
                    HStack {
                        ColorButton(color: .white, isOn: Binding(get: { theme == .default }, set: { _, _ in theme = .default }))
                        ColorButton(color: .systemPink, isOn: Binding(get: { theme == .pink }, set: { _, _ in theme = .pink }))
                        ColorButton(color: .secondaryLabel, isOn: Binding(get: { theme == .monochrome }, set: { _, _ in theme = .monochrome }))
                    }

                    VStack {
                        Toggle("Atmosphere Use Theme", isOn: $atmosphereUseTheme)
                        Toggle("Circle Use Theme", isOn: $circleUseTheme)
                    }
                }
                .floating()
            }
            .padding(.bottom, 30)
        }
    }
}

private struct ColorButton: View {
    let color1: UIColor
    let color2: UIColor
    let isOn: Binding<Bool>

    init(color: UIColor, isOn: Binding<Bool>) {
        self.color1 = color
        self.color2 = color
        self.isOn = isOn
    }

    init(color1: UIColor, color2: UIColor, isOn: Binding<Bool>) {
        self.color1 = color1
        self.color2 = color2
        self.isOn = isOn
    }

    var body: some View {
        Button {
            isOn.wrappedValue.toggle()
        } label: {
            ZStack {
                Circle()
                    .fill(
                        LinearGradient(
                            gradient: Gradient(colors: [Color(color1), Color(color2)]),
                            startPoint: .leading,
                            endPoint: .trailing
                        )
                    )
                Circle().strokeBorder(Color(color1.darker), lineWidth: 2)
            }
        }
        .opacity(isOn.wrappedValue ? 1.0 : 0.2)
        .frame(width: 50, height: 50)
    }
}

private struct TestLayer: MapStyleContent {
    var id: String
    var radius: LocationDistance
    var color: UIColor
    var coordinate: CLLocationCoordinate2D
    var useTheme: Bool

    var body: some MapStyleContent {
        let sourceId = "\(id)-source"
        FillLayer(id: id, source: sourceId)
            .fillColorUseTheme(useTheme ? .default : .none)
            .fillColor(color)
            .fillOpacity(0.4)
        LineLayer(id: "\(id)-border", source: sourceId)
            .lineColor(color.darker)
            .lineColorUseTheme(useTheme ? .default : .none)
            .lineOpacity(0.4)
            .lineWidth(2)
        GeoJSONSource(id: sourceId)
            .data(.geometry(.polygon(Polygon(center: coordinate, radius: radius * 1000000, vertices: 60))))
    }
}

private let styleURL = Bundle.main.url(forResource: "fragment-realestate-NY", withExtension: "json")!
private let monochromeTheme = UIImage(named: "monochrome_lut")!
private let pinkTheme = "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"

struct ColorThemeExample_Previews: PreviewProvider {
    static var previews: some View {
        StandardStyleImportExample()
    }
}
